<?php
/**
 * Portfolio custom post type.
 */

// Prevent loading this file directly
defined( 'ABSPATH' ) || exit;

class Erado_Addons_Portfolio {
	/**
	 * Construct function.
	 *
	 * @return  void
	 */
	function __construct() {
		add_action( 'init', array( __CLASS__, 'portfolio_init' ) );
		add_filter( 'single_template', array( $this, 'portfolio_single' ) );
		add_filter( 'archive_template', array( $this, 'portfolio_archive' ) );
	}

	/**
	 * Register a portfolio post type.
	 *
	 * @link http://codex.wordpress.org/Function_Reference/register_post_type
	 */
	public static function portfolio_init() {
		register_post_type( 'portfolio',
			array(
				'public'             => true,
				'publicly_queryable' => true,
				'show_ui'            => true,
				'show_in_menu'       => true,
				'query_var'          => true,
				'rewrite'            => array( 'slug' => 'portfolio' ),
				'capability_type'    => 'post',
				'has_archive'        => true,
				'hierarchical'       => false,
				'menu_position'      => 99,
				'menu_icon'          => 'dashicons-filter',
				'supports'           => array( 'title', 'editor', 'thumbnail' ),
				'labels'             => array(
					'name'               => _x( 'Portfolio', 'erado-addons' ),
					'singular_name'      => _x( 'Portfolio', 'erado-addons' ),
					'menu_name'          => _x( 'Portfolio', 'erado-addons' ),
					'name_admin_bar'     => _x( 'Portfolio', 'erado-addons' ),
					'add_new'            => _x( 'Add New', 'erado-addons' ),
					'add_new_item'       => __( 'Add New Portfolio', 'erado-addons' ),
					'new_item'           => __( 'New Portfolio', 'erado-addons' ),
					'edit_item'          => __( 'Edit Portfolio', 'erado-addons' ),
					'view_item'          => __( 'View Portfolio', 'erado-addons' ),
					'all_items'          => __( 'All Portfolios', 'erado-addons' ),
					'search_items'       => __( 'Search Portfolios', 'erado-addons' ),
					'parent_item_colon'  => __( 'Parent Portfolios:', 'erado-addons' ),
					'not_found'          => __( 'No portfolios found.', 'erado-addons' ),
					'not_found_in_trash' => __( 'No portfolios found in Trash.', 'erado-addons' )
				),
			)
		);

		// Register portfolio category
		register_taxonomy( 'portfolio_cat',
			array( 'portfolio' ),
			array(
				'hierarchical'      => true,
				'show_ui'           => true,
				'show_admin_column' => true,
				'query_var'         => true,
				'rewrite'           => array( 'slug' => 'portfolio_cat' ),
				'labels'            => array(
					'name'              => _x( 'Categories', 'erado-addons' ),
					'singular_name'     => _x( 'Category', 'erado-addons' ),
					'search_items'      => __( 'Search Categories', 'erado-addons' ),
					'all_items'         => __( 'All Categories', 'erado-addons' ),
					'parent_item'       => __( 'Parent Category', 'erado-addons' ),
					'parent_item_colon' => __( 'Parent Category:', 'erado-addons' ),
					'edit_item'         => __( 'Edit Category', 'erado-addons' ),
					'update_item'       => __( 'Update Category', 'erado-addons' ),
					'add_new_item'      => __( 'Add New Category', 'erado-addons' ),
					'new_item_name'     => __( 'New Category Name', 'erado-addons' ),
					'menu_name'         => __( 'Categories', 'erado-addons' ),
				),
			)
		);

		// Register portfolio project client
		register_taxonomy( 'portfolio_client',
			'portfolio',
			array(
				'hierarchical'          => true,
				'show_ui'               => true,
				'show_admin_column'     => true,
				'query_var'             => true,
				'rewrite'               => array( 'slug' => 'portfolio_client' ),
				'labels'                => array(
					'name'                       => _x( 'Clients', 'erado-addons' ),
					'singular_name'              => _x( 'Client', 'erado-addons' ),
					'search_items'               => __( 'Search Clients', 'erado-addons' ),
					'all_items'                  => __( 'All Clients', 'erado-addons' ),
					'parent_item'                => null,
					'parent_item_colon'          => null,
					'edit_item'                  => __( 'Edit Client', 'erado-addons' ),
					'update_item'                => __( 'Update Client', 'erado-addons' ),
					'add_new_item'               => __( 'Add New Client', 'erado-addons' ),
					'new_item_name'              => __( 'New Client Name', 'erado-addons' ),
					'separate_items_with_commas' => __( 'Separate writers with commas', 'erado-addons' ),
					'add_or_remove_items'        => __( 'Add or remove writers', 'erado-addons' ),
					'choose_from_most_used'      => __( 'Choose from the most used writers', 'erado-addons' ),
					'not_found'                  => __( 'No writers found.', 'erado-addons' ),
					'menu_name'                  => __( 'Clients', 'erado-addons' ),
				),
			)
		);

		// Register portfolio tag
		register_taxonomy( 'portfolio_tag',
			'portfolio',
			array(
				'hierarchical'          => false,
				'show_ui'               => true,
				'show_admin_column'     => true,
				'update_count_callback' => '_update_post_term_count',
				'query_var'             => true,
				'rewrite'               => array( 'slug' => 'portfolio_tag' ),
				'labels'                => array(
					'name'                       => _x( 'Tags', 'erado-addons' ),
					'singular_name'              => _x( 'Tag', 'erado-addons' ),
					'search_items'               => __( 'Search Tags', 'erado-addons' ),
					'popular_items'              => __( 'Popular Tags', 'erado-addons' ),
					'all_items'                  => __( 'All Tags', 'erado-addons' ),
					'parent_item'                => null,
					'parent_item_colon'          => null,
					'edit_item'                  => __( 'Edit Tag', 'erado-addons' ),
					'update_item'                => __( 'Update Tag', 'erado-addons' ),
					'add_new_item'               => __( 'Add New Tag', 'erado-addons' ),
					'new_item_name'              => __( 'New Tag Name', 'erado-addons' ),
					'separate_items_with_commas' => __( 'Separate writers with commas', 'erado-addons' ),
					'add_or_remove_items'        => __( 'Add or remove writers', 'erado-addons' ),
					'choose_from_most_used'      => __( 'Choose from the most used writers', 'erado-addons' ),
					'not_found'                  => __( 'No writers found.', 'erado-addons' ),
					'menu_name'                  => __( 'Tags', 'erado-addons' ),
				),
			)
		);
	}

	/**
	 * Load single item template file for the portfolio custom post type.
	 *
	 * @param   string  $template  Current template file.
	 *
	 * @return  string
	 */
	function portfolio_single( $template ) {
		global $post;

		if ( $post->post_type == 'portfolio' ) {
			$template = ERADO_ADDONS_PATH . 'inc/portfolio/views/single.php';
		}

		return $template;
	}

	/**
	 * Load archive template file for the portfolio custom post type.
	 *
	 * @param   string  $template  Current template file.
	 *
	 * @return  string
	 */
	function portfolio_archive( $template ) {
		global $post;

		if ( isset( $post->post_type ) && $post->post_type == 'portfolio' ) {
			$template = ERADO_ADDONS_PATH . 'inc/portfolio/views/archive.php';
		}

		return $template;
	}

	public static function related() {
		global $post;

		// Get the portfolio tags.
		$tags = get_the_terms( $post, 'portfolio_tag' );

		if ( $tags ) {
			$tag_ids = array();

			foreach ( $tags as $tag ) {
				$tag_ids[] = $tag->term_id;
			}

			$args = array(
				'post_type'      => 'portfolio',
				'post__not_in'   => array( $post->ID ),
				'posts_per_page' => -1,
				'tax_query'      => array(
					array(
						'taxonomy' => 'portfolio_tag',
						'field'    => 'id',
						'terms'    => $tag_ids,
					),
				)
			);

			// Get portfolio category
			$categories = wp_get_post_terms( get_the_ID(), 'portfolio_cat' );

			$the_query = new WP_Query( $args );
			?>
			<div class="container mt_50 pt_50 related-portfolio">
				<div class="addon-title mb_40">
					<h3><?php echo esc_html__( 'Related Portfolio', 'erado-addons' ); ?></h3>
				</div>
				<div class="related-carousel owl-theme owl-carousel" data-carousel='{"selector":".related-carousel", "itemDesktop": 4, "itemMedium": 3, "itemTablet": 2, "itemMobile": 1, "margin":30 , "nav":true}'>
					<?php while ( $the_query->have_posts() ) : $the_query->the_post(); ?>
						<div id="portfolio-<?php the_ID(); ?>" class="portfolio-item">
							<?php if ( has_post_thumbnail() ) : ?>
								<a class="db mb_15" href="<?php esc_url( the_permalink() ); ?>">
									<?php
										$img = wp_get_attachment_image_src( get_post_thumbnail_id( get_the_ID() ), 'full' );
											$image = aq_resize( $img[0], 450, 450, true ); ?>
											<img src="<?php echo esc_url( $image ); ?>" width="450" height="450" alt="<?php the_title(); ?>" />
										<?php
									?>
								</a>
							<?php endif; ?>
							<div class="portfolio-info">
								<h4><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h4>
								<?php
									if ( $categories ) {
										echo '<span>' . get_the_term_list( $post->ID, 'portfolio_cat', '', ' / ' ) . '</span>';
									}
								?>
							</div>
						</div>
					<?php endwhile; ?>
				</div>
			</div>
		<?php
		}

		wp_reset_postdata();
	}
}
$portfolio = new Erado_Addons_Portfolio;
